;;;
;;; asm6502.lisp
;;;
;;; Assemble machine code for 6502 processor.
;;;
;;; Joseph A. Oswald, III <josephoswald@gmail.com>
;;; 
;;; $Version$
;;;
;;; $Log$
;;;

#|
;; example

  (asm6502 (:org #x801 ...other options)
    (ldy :immediate 0)
    label1
    (lda :absolute-y str)
    (beq exit) ; branch relative is default
    (jsr :absolute cout) ; cout label defined elsewhere
    (iny) ; implied is default
    (bne label1)
    exit
    (rts)
    str
    (cstring "Message" :high)
)

--> 
#x801 
#(160 0 ; 801: ldy #0 
  185 #xyy #x08 ; 803: lda str,y
  240 #x06 ; 806: bne exit 
  #x20 #ed #xfd ; jsr $fded
   200 ; iny
   208 245 ; bne label1
   #x60
   205 ; "M", high ascii
   ....
   229 ; #\e, high ascii
   0
   )
((label1 . #x803) (exit . #x8xx) (str . #x8yy) 

|#

#|
(translate-accum-mode ()
	   (if (eql address-mode :accumulator)
	       :implied
	       address-mode))
|#

(eval-when (:compile-toplevel :load-toplevel :execute)
  (defun make-mnemonic-hash (opcode-list)
    "Takes a list of opcodes, specified in the form
   (numeric-opcode :OP :ADDRESS-MODE) and returns a
   hash table, where the key :OP hashes to an 
   association list with :address-mode as key returning
   (:address-mode . numeric-opcode)"
    
    (let ((hash (make-hash-table)))
      (mapcar #'(lambda (opcode-elt)
		  (let ((opcode (first opcode-elt))
			(mnemonic (second opcode-elt))
			(address-mode (third opcode-elt)))
		    (setf (gethash mnemonic hash)
			  (acons address-mode opcode (gethash mnemonic hash)))))
	      opcode-list)
      hash)))

(defconstant +6502-mnemonic-hash+
  (make-mnemonic-hash +6502-opcode-list+))

(define-condition bad-address-mode (error)
  ((mnemonic :reader mnemonic :initarg :mnemonic)
   (address-mode :reader address-mode :initarg :address-mode))
  (:report (lambda (condition stream)
	     (format stream "~A does not support ~A address mode."
		     (mnemonic condition)
		     (address-mode condition)))))

(define-condition unspecified-address-mode (error)
  ((mnemonic :reader mnemonic :initarg :mnemonic))
  (:report (lambda (condition stream)
	     (format stream "Must specify addressing mode for ~A."
		     (mnemonic condition)))))

(defun asm-6502-lookup (mnemonic &optional address-mode)
  "Returns multiple values:
    numeric opcode (or nil if mnemonic not found in table)
    address-mode 
    instruction-length (in bytes).

   If address-mode is omitted (nil), there must be only a single
allowed address mode, or an error 'unspecified-address-mode is signaled.
   If address-mode is specified, and the mode is not applicable to the
mnemonic, an error 'bad-address-mode will be signaled."

  (multiple-value-bind (opcode-assoc found)
      (gethash mnemonic +6502-mnemonic-hash+)
    (if found
	(let ((opcode (if address-mode
			  (cdr (assoc address-mode opcode-assoc))
			  (if (= 1 (length opcode-assoc))
			      (let ((default (first opcode-assoc)))
				(setf address-mode (car default))
				(cdr default))
			      (error 'unspecified-address-mode :mnemonic
				     mnemonic)))))
	  (if opcode
	      (values opcode address-mode (6502-instruction-length opcode))
	      (error 'bad-address-mode :mnemonic mnemonic
		     :address-mode address-mode)))
	(values nil 0))))
  
;;; assembler
;;;
;;; first pass: 
;;; determine labels, verify address modes, expand data pseudo-ops
;;;

(defvar *current-instruction* 0)

(defvar *pseudo-ops* (make-hash-table))
(defvar *label-hash* (make-hash-table))

;; DARN: need first and second pass actions, because
;; labels needed for arithmetic cannot be calculated until the 
;; second pass, but we need to know lengths for the second pass

;; OK, I've screwed up the EVAL-ARG function by leaving expressions in labels
;; where they can't be evaluated yet. 

(eval-when (:compile-toplevel :load-toplevel :execute)
  (defmacro defasmop (opcode &key first-pass second-pass)
    "defasmop should contain a body, which constructs an (:emit ...) 
list based on the ARGS list that is the CDR of the pseudo-op."
    (let ((sym (intern (symbol-name opcode) :keyword)))
      `(setf (gethash ,sym *pseudo-ops*)
      (cons 
       (lambda (args) ,first-pass)
       (lambda (args) ,second-pass))))))
  
(defun eval-math-as-possible (expr)
  (cond 
    ((symbolp expr)
     (multiple-value-bind (value found)
	 (gethash expr *label-hash*)
       (if found 
	   value
	   (if (boundp expr)
	       (symbol-value expr)
	       expr))))
    ((consp expr)
     (let ((funsym (car expr))
	   (args (mapcar #'eval-math-as-possible (cdr expr))))
       (if (and (fboundp funsym)
		(notany #'symbolp args))
	   (apply (symbol-function funsym) args)
	   (cons funsym args))))
    (t expr)))

(defasmop EQU 
    :first-pass
  (progn
    (unless (= 2 (length ARGS))
      (error "EQU must have 2 arguments"))
    (let ((symbol (first ARGS))
	  (val-expr (second ARGS)))
      (format t "EQU ~A ~A~%" symbol (cdr ARGS))
      (multiple-value-bind (value found)
	  (gethash symbol *label-hash*)
	(when found (error "Redefining value of label ~A" symbol))
	(let ((val (eval-math-as-possible val-expr)))
	  (setf (gethash symbol *label-hash*) 
		val)
	  (list :EQU symbol val)))))
  :second-pass
  (let ((symbol (first ARGS))
	(expr (second ARGS))) 
    (setf (gethash symbol *label-hash*)
	  (eval-arg expr *label-hash*))
    nil))

(defasmop DS
    :first-pass
  (progn
    (unless (= 1 (length ARGS))
      (error "DS takes only one argument"))
    (let ((length (eval-arg (first ARGS) *label-hash*)))
      (incf *current-instruction* length)
      (list :emit-space (eval-arg (first ARGS) *label-hash*))))
  :second-pass
  (error "DS must be resolved on first pass."))

(defasmop DB
    :first-pass
  (progn
    (incf *current-instruction* (length ARGS))
    (cons :db (mapcar #'eval-math-as-possible ARGS)))
  :second-pass
  (let* ((len 0)
	 (val-list (mapcar #'(lambda (expr) 
			       (let ((val (eval-arg expr *label-hash*)))
				 (unless (< -128 val 256)
				   (error "Value ~A exceeds byte range"
					  expr))
				 (incf len)
				 val))
			   ARGS)))
    (incf *current-instruction* len)
    (cons :emit val-list)))

(defasmop ORG
    :first-pass
    (progn
      (unless (= 1 (length ARGS))
	(error "ORG takes only one argument"))
      (setf *current-instruction* (eval-arg (car ARGS) *label-hash*)))
    :second-pass
    (setf *current-instruction* (eval-arg (car ARGS) *label-hash*)))
    
(defun process-pseudo-op-first (opcode rest)
  (unless (keywordp opcode)
    (error "Pseudo-op ~A must be converted to a keyword for lookup"
	   opcode))
  (multiple-value-bind (opfunction found)
      (gethash opcode *pseudo-ops*)
    (unless found
      (error "Pseudo-op ~A undefined" opcode))
    (let ((result (funcall (car opfunction) rest)))
      (if result
	  result
	  (cons opcode rest)))))

(defun process-pseudo-op-second (opcode rest)
  (multiple-value-bind (opfunction found)
      (gethash opcode *pseudo-ops*)
    (unless found
      (error "Pseudo-op ~A undefined" opcode))
    (funcall (cdr opfunction) rest)))

(defun process-first-pass-elt (elt)
  (if (symbolp elt)
      (multiple-value-bind (value found) (gethash elt *label-hash*)
	(if found
	    (error "Multiple use of label ~S" elt)
	    (setf (gethash elt *label-hash*) *current-instruction*)))
      (progn
	(unless (consp elt)
	  (error "Bad atom ~S" elt))
	(unless (symbolp (first elt))
	  (error "Non-symbolic opcode ~S" (first elt)))
	(let ((opcode (intern (symbol-name (first elt)) :keyword))
	      (l (length elt)))    
	  ;; (opcode) ; only for implied, or pseudo-op
	  ;; (opcode address-mode) ; for implied, accumulator
	  ;; (opcode arg) ; only for opcodes with default address mode
	  ;; (opcode address-mode arg) 
	  (let ((specified-address-mode (if (and (>= l 2) (keywordp (second 
								     elt)))
					    (second elt)
					    nil)))
	    (multiple-value-bind (num-opcode address-mode
					     instruction-length)
		(asm-6502-lookup opcode specified-address-mode)
	    (if num-opcode
		(progn
		  (incf *current-instruction* instruction-length)
		  (list* opcode address-mode
			 (if specified-address-mode
			     (cddr elt)
			     (cdr elt))))
		(process-pseudo-op-first opcode (cdr elt)))))))))

(defun first-pass (code-list start-address)
  (setf *current-instruction* start-address)
  (setf *label-hash* (make-hash-table))
  (mapcar #'process-first-pass-elt code-list))

(defun eval-arg (arg-expr symbol-table)
  (let ((syms nil)
	(vals nil))
    (maphash #'(lambda (key value) 
		 (push key syms)
		 (push value vals))
	     symbol-table)
    (progv syms vals
      (eval arg-expr))))
#|
  (cond
    ((symbolp arg-expr)
     (multiple-value-bind (value found)
	 (gethash arg-expr symbol-table)
       (if found 
	   value
	   (symbol-value arg-expr)))
     (consp arg-expr)
     ; Lisp special forms need bindings
     ; Macros could be expanded
     ; Normal functions need bindings
  )
|#

#|
;;; test cases
;;;

(defun dump-symbols ()
  (maphash #'(lambda (key value)
	       (format t "~A = ~D~%" key value))
	   *label-hash*))

(setf *first-pass-result* nil)

(setq *first-pass-result*
  (first-pass '(start 
		(equ bar 10)
		(equ baz (+ bar 10))
		(equ frob (+ boz 20))
		(equ boz 30)
		(lda :immediate 0) 
		(sta :absolute var) 
		(rts) 
		var 
		(ds 10)
		var2
		(db 1 2 (- baz bar) 0)
		var3)
	      #x800))

;; FIXME: how to get FROB resolved (+ boz 20) -> 50

(eval-arg 0 *label-hash*)
(eval-arg 'var *label-hash*)
(eval-arg '0 *label-hash*)
(eval-arg 'start *label-hash*)

(eval-arg start *label-hash*)

|#
		 
